package com.blocksignature.signature;


import co.nstant.in.cbor.CborBuilder;
import co.nstant.in.cbor.CborEncoder;
import co.nstant.in.cbor.CborException;
import co.nstant.in.cbor.model.UnsignedInteger;
import com.blocksignature.entity.Address;
import com.blocksignature.entity.Payload;
import com.blocksignature.entity.Secp256k1;
import com.blocksignature.signature.*;
import org.web3j.crypto.ECKeyPair;
import org.web3j.utils.Numeric;
import ove.crypto.digest.Blake2b;

import java.io.ByteArrayOutputStream;
import java.math.BigInteger;
import java.util.Base64;

public class SignatureTransactionUtil {
    /**
     * 形成摘要需要拼接的字符串
     */
    public static byte[] CID_PREFIX = new byte[]{0x01, 0x71, (byte) 0xa0, (byte) 0xe4, 0x02, 0x20};

    /**
     * @param message 交易结构体的序列化字节
     *                通过交易结构体字节获取CidHash
     */
    public static byte[] getCidHash(byte[] message) {
        Blake2b.Param param = new Blake2b.Param();
        param.setDigestLength(32);

        //消息体字节
        byte[] messageByte = Blake2b.Digest.newInstance(param).digest(message);

        int xlen = CID_PREFIX.length;
        int ylen = messageByte.length;

        byte[] result = new byte[xlen + ylen];

        System.arraycopy(CID_PREFIX, 0, result, 0, xlen);
        System.arraycopy(messageByte, 0, result, xlen, ylen);

        byte[] prefixByte = Blake2b.Digest.newInstance(param).digest(result);
        return prefixByte;
    }

    /**
     * @param cidHash 摘要
     *                对摘要进行椭圆签名椭圆签名
     */
    public static String sign(byte[] cidHash, String privateKey) {
        ECKeyPair ecKeyPair = ECKeyPair.create(Numeric.toBigInt(privateKey));
        org.web3j.crypto.Sign.SignatureData signatureData = org.web3j.crypto.Sign.signMessage(cidHash,
                ecKeyPair, false);
        byte[] sig = getSignature(signatureData);
        String base64 = Base64.getEncoder().encodeToString(sig);
        return base64;
    }

    /**
     * 获取签名
     *
     * @param signatureData
     * @return
     */
    private static byte[] getSignature(org.web3j.crypto.Sign.SignatureData signatureData) {
        byte[] sig = new byte[65];
        System.arraycopy(signatureData.getR(), 0, sig, 0, 32);
        System.arraycopy(signatureData.getS(), 0, sig, 32, 32);
        sig[64] = (byte) ((signatureData.getV() & 0xFF) - 27);//为啥减去27看signMessage（）方法（内部源码）这面用的web3j的签名库，web3j的签名对recId加了27，所以这面要减去拿到原v
        return sig;
    }

//    public static void main(String[] args) {
//
//        UnsignedMessageAPI messageAPI = new UnsignedMessageAPI();
//        messageAPI.setFrom("f1oiirx7jwajfi34gmkqio2wysbyvprgl273qrinq");
//        messageAPI.setGas_limit(898989);
//        messageAPI.setGasFeeCap("98989800");
//        messageAPI.setGasPremium("909090");
//        messageAPI.setMethod(0);
//        messageAPI.setTo("f1dpr2or4qy7vytin244noqgmrrb6isegmvwb3g3a");
//        messageAPI.setValue("10000000000000000000");
//        messageAPI.setParams("");
//        String privateKey = "5909fb85f650879c51113a2aaea89b51c849a11cfb7870c24e7ebfd325607ff1";
//        String result = transaction_serialize(messageAPI,privateKey);
//        System.out.println(result);
//    }

    /**
     * @param unsignedMessageAPI
     */
    public static String transaction_serialize(UnsignedMessageAPI unsignedMessageAPI, String privateKey) {

        UnsignedMessage unsignedMessage = try_from(unsignedMessageAPI);
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try {
            new CborEncoder(baos).encode(new CborBuilder()
                    .addArray()
                    .add(unsignedMessage.getVersion())
                    // add string
                    .add(unsignedMessage.getTo())
                    .add(unsignedMessage.getFrom())
                    .add(unsignedMessage.getSequence())
                    .add(unsignedMessage.getValue())
                    .add(unsignedMessage.getGas_limit())
                    .add(unsignedMessage.getGasFeeCap())
                    .add(unsignedMessage.getGasPremium())
                    .add(unsignedMessage.getMethod_num())
                    // add integer
                    .add(new co.nstant.in.cbor.model.ByteString(new byte[]{}))
                    .end()
                    .build());
            byte[] encodedBytes = baos.toByteArray();
            byte[] cidHashBytes = getCidHash(encodedBytes);
            return sign(cidHashBytes, privateKey);
        } catch (CborException e) {
            e.printStackTrace();
        }
        return null;
    }

    public static UnsignedMessage try_from(UnsignedMessageAPI unsignedMessageAPI) {

        //构建交易结构体
        Address from = from_str(unsignedMessageAPI.getFrom());
        Address to = from_str(unsignedMessageAPI.getTo());
        UnsignedMessage unsignedMessage = new UnsignedMessage();
        unsignedMessage.setVersion(new UnsignedInteger(0));
        unsignedMessage.setTo(new co.nstant.in.cbor.model.ByteString(to.getPayload().getSecp256k1().getBytes()));

        unsignedMessage.setFrom(new co.nstant.in.cbor.model.ByteString(from.getPayload().getSecp256k1().getBytes()));

        unsignedMessage.setSequence(new UnsignedInteger(unsignedMessageAPI.getNonce()));
        co.nstant.in.cbor.model.ByteString valueByteString = null;
        if (new BigInteger(unsignedMessageAPI.getValue()).toByteArray()[0] != 0) {
            byte[] byte1 = new byte[new BigInteger(unsignedMessageAPI.getValue()).toByteArray().length + 1];
            byte1[0] = 0;
            System.arraycopy(new BigInteger(unsignedMessageAPI.getValue()).toByteArray(), 0, byte1, 1, new BigInteger(unsignedMessageAPI.getValue()).toByteArray().length);
            valueByteString = new co.nstant.in.cbor.model
                    .ByteString(byte1);
        } else {
            valueByteString = new co.nstant.in.cbor.model
                    .ByteString(new BigInteger(unsignedMessageAPI.getValue()).toByteArray());
        }

        unsignedMessage.setValue(valueByteString);
        unsignedMessage.setGas_limit(new UnsignedInteger(unsignedMessageAPI.getGas_limit()));

        co.nstant.in.cbor.model.ByteString gasFeeCapString = null;
        if (new BigInteger(unsignedMessageAPI.getGasFeeCap()).toByteArray()[0] != 0) {
            byte[] byte2 = new byte[new BigInteger(unsignedMessageAPI.getGasFeeCap()).toByteArray().length + 1];
            byte2[0] = 0;
            System.arraycopy(new BigInteger(unsignedMessageAPI.getGasFeeCap()).toByteArray(), 0, byte2, 1
                    , new BigInteger(unsignedMessageAPI.getGasFeeCap()).toByteArray().length);
            gasFeeCapString = new co.nstant.in.cbor.model
                    .ByteString(byte2);
        } else {
            gasFeeCapString = new co.nstant.in.cbor.model
                    .ByteString(new BigInteger(unsignedMessageAPI.getGasFeeCap()).toByteArray());
        }

        unsignedMessage.setGasFeeCap(gasFeeCapString);


        co.nstant.in.cbor.model.ByteString gasGasPremium = null;
        if (new BigInteger(unsignedMessageAPI.getGasPremium()).toByteArray()[0] != 0) {
            byte[] byte2 = new byte[new BigInteger(unsignedMessageAPI.getGasPremium()).toByteArray().length + 1];
            byte2[0] = 0;
            System.arraycopy(new BigInteger(unsignedMessageAPI.getGasPremium()).toByteArray(), 0, byte2, 1
                    , new BigInteger(unsignedMessageAPI.getGasPremium()).toByteArray().length);
            gasGasPremium = new co.nstant.in.cbor.model
                    .ByteString(byte2);
        } else {
            gasGasPremium = new co.nstant.in.cbor.model
                    .ByteString(new BigInteger(unsignedMessageAPI.getGasPremium()).toByteArray());
        }

        unsignedMessage.setGasPremium(gasGasPremium);


        unsignedMessage.setMethod_num(new UnsignedInteger(0));
        unsignedMessage.setParams(new co.nstant.in.cbor.model.ByteString(new byte[0]));
        return unsignedMessage;
    }


    public static Address from_str(String addressStr) {
        Address address = new Address();
        //去掉前两位
        String str = addressStr.substring(2);

        byte[] bytes12 = new byte[21];

        //为啥加1，因为是Secp256k1的标识就是1
        bytes12[0] = 1;
        System.arraycopy(Base32.decode(str), 0, bytes12, 1, 20);
        Secp256k1 secp256k1 = new Secp256k1();
        secp256k1.setBytes(bytes12);
        Payload payload = new Payload();
        payload.setSecp256k1(secp256k1);
        address.setPayload(payload);
        return address;
    }
}
