package fil

import (
	"encoding/base64"
	"encoding/hex"
	"encoding/json"
	"errors"
	"fmt"
	"github.com/btcsuite/btcd/btcec"
	"github.com/btcsuite/btcd/chaincfg"
	"github.com/btcsuite/btcutil"
	"github.com/multiformats/go-varint"
	"github.com/yancaitech/go-filutils"
	_ "go-hodler-keys-master/src"
	keys "keys/src"
)

const defaultMethod = 0

func FilecoinSignRawTx(wifPriKey string, nonce uint64, fromAccount string, toAccount string, value uint64, gasPrice uint64, gaslimit uint64) (signedtx string, hextx string, txid string, err error) {
	var key keys.Key
	key.LoadBitcoinWIF(wifPriKey)
	if err != nil {
		return "", "", "", err
	}
	fromAddr, err := key.FilecoinAddress()
	if err != nil {
		return "", "", "", err
	}
	if fromAccount != fromAddr {
		return "", "", "", errors.New("address not match key")
	}

	hexkey, err := key.DumpBitcoinHex()
	if err != nil {
		return "", "", "", err
	}
	hexstr := hex.EncodeToString(hexkey)
	pk, err := filutils.LoadFromPrivateKey(hexstr)
	if err != nil {
		return "", "", "", err
	}
	sk, err := pk.DumpPrivateKey()
	if err != nil {
		return "", "", "", err
	}

	raw, err := filutils.CreateTransaction(fromAccount,
		toAccount,
		filutils.NewIntUnsigned(value),
		int64(gaslimit),
		filutils.NewIntUnsigned(gasPrice),
		nonce,
		defaultMethod,
		nil)
	if err != nil {
		return "", "", "", err
	}

	tx, err := filutils.DecodeTransaction(raw)
	if err != nil {
		return "", "", "", err
	}

	stx, err := filutils.SignMessage(sk, tx)
	if err != nil {
		return "", "", "", err
	}

	bs, err := json.MarshalIndent(stx, "", "  ")
	if err != nil {
		return "", "", "", err
	}
	signedtx = string(bs)

	bs, err = stx.Serialize()
	if err != nil {
		return "", "", "", err
	}
	hextx = hex.EncodeToString(bs)

	txid = stx.Cid().String()

	return signedtx, hextx, txid, nil
}

func FilecoinAddressByPubKey(pubKeyBase64 string) (addr string, err error) {
	base64er := base64.RawStdEncoding
	decoder_buf, _ := base64er.DecodeString(pubKeyBase64)
	pk, _ := btcec.ParsePubKey(decoder_buf, btcec.S256())
	var address filutils.Address
	address, err = filutils.NewSecp256k1Address(pk.SerializeCompressed())
	addrEncode, _ := encode(filutils.Mainnet, address)
	return addrEncode, nil
}

// BitcoinAddress func
func BitcoinAddress(pubKeyBase64 string) (addr string, err error) {
	base64er := base64.RawStdEncoding
	decoder_buf, _ := base64er.DecodeString(pubKeyBase64)
	pk, _ := btcec.ParsePubKey(decoder_buf, btcec.S256())
	var params *chaincfg.Params
	params = &chaincfg.MainNetParams
	pkadr, err := btcutil.NewAddressPubKey(pk.SerializeCompressed(), params)
	if err != nil {
		return
	}
	pkh := pkadr.AddressPubKeyHash()
	addr = pkh.EncodeAddress()

	return addr, nil
}

func encode(network filutils.Network, addr filutils.Address) (string, error) {
	if addr == filutils.Undef {
		return filutils.UndefAddressString, nil
	}
	var ntwk string
	switch network {
	case filutils.Mainnet:
		ntwk = filutils.MainnetPrefix
	case filutils.Testnet:
		ntwk = filutils.TestnetPrefix
	}

	var strAddr string
	switch addr.Protocol() {
	case filutils.SECP256K1, filutils.Actor, filutils.BLS:
		cksm := filutils.Checksum(append([]byte{addr.Protocol()}, addr.Payload()...))
		strAddr = ntwk + fmt.Sprintf("%d", addr.Protocol()) + filutils.AddressEncoding.WithPadding(-1).EncodeToString(append(addr.Payload(), cksm[:]...))
	case filutils.ID:
		i, n, err := varint.FromUvarint(addr.Payload())
		if err != nil {
			return filutils.UndefAddressString, err
		}
		if n != len(addr.Payload()) {
			return filutils.UndefAddressString, err
		}
		strAddr = fmt.Sprintf("%s%d%d", ntwk, addr.Protocol(), i)
	}
	return strAddr, nil
}
