package keys

import (
	"encoding/base64"
	"encoding/hex"
	"encoding/json"
	"github.com/btcsuite/btcd/btcec"
	"github.com/btcsuite/btcd/chaincfg"
	"github.com/btcsuite/btcutil"
	"keys/filutils"
)

const defaultMethod = 0

func FilecoinSignRawTx(wifPriKey string, nonce uint64, toAccount string, value uint64, gasPremium uint64, gasFeeCap uint64, gaslimit uint64) (signedtx string, hextx string, txid string, err error) {
	var key Key
	key.LoadBitcoinWIF(wifPriKey)
	if err != nil {
		return "", "", "", err
	}
	fromAccount, err := key.FilecoinAddress()
	if err != nil {
		return "", "", "", err
	}

	hexkey, err := key.DumpBitcoinHex()
	if err != nil {
		return "", "", "", err
	}
	hexstr := hex.EncodeToString(hexkey)
	pk, err := filutils.LoadFromPrivateKey(hexstr)
	if err != nil {
		return "", "", "", err
	}
	sk, err := pk.DumpPrivateKey()
	if err != nil {
		return "", "", "", err
	}

	raw, err := filutils.CreateTransaction(fromAccount,
		toAccount,
		filutils.NewIntUnsigned(value),
		int64(gaslimit),
		filutils.NewIntUnsigned(gasPremium),
		filutils.NewIntUnsigned(gasFeeCap),
		nonce,
		defaultMethod,
		nil)
	if err != nil {
		return "", "", "", err
	}

	tx, err := filutils.DecodeTransaction(raw)
	if err != nil {
		return "", "", "", err
	}

	stx, err := filutils.SignMessage(sk, tx)
	if err != nil {
		return "", "", "", err
	}

	bs, err := json.MarshalIndent(stx, "", "  ")
	if err != nil {
		return "", "", "", err
	}
	signedtx = string(bs)

	bs, err = stx.Serialize()
	if err != nil {
		return "", "", "", err
	}
	hextx = hex.EncodeToString(bs)

	txid = stx.Cid().String()

	return signedtx, hextx, txid, nil
}

func FilecoinAddressByPubKey(pubKeyBase64 string) (addr string, err error) {
	base64er := base64.RawStdEncoding
	decoder_buf, _ := base64er.DecodeString(pubKeyBase64)
	pk, _ := btcec.ParsePubKey(decoder_buf, btcec.S256())
	var address filutils.Address
	address, err = filutils.NewSecp256k1Address(pk.SerializeCompressed())
	addrEncode, _ := filutils.Encode(filutils.Mainnet, address)
	return addrEncode, nil
}

// BitcoinAddress func
func BitcoinAddress(pubKeyBase64 string) (addr string, err error) {
	base64er := base64.RawStdEncoding
	decoder_buf, _ := base64er.DecodeString(pubKeyBase64)
	pk, _ := btcec.ParsePubKey(decoder_buf, btcec.S256())
	var params *chaincfg.Params
	params = &chaincfg.MainNetParams
	pkadr, err := btcutil.NewAddressPubKey(pk.SerializeCompressed(), params)
	if err != nil {
		return
	}
	pkh := pkadr.AddressPubKeyHash()
	addr = pkh.EncodeAddress()

	return addr, nil
}
